// Flluir Premiere Plugin - ExtendScript (JSX)
// Este arquivo contém funções ExtendScript para interagir com o Adobe Premiere Pro

(function(thisObj) {
  'use strict';

  // NOTA: Não verificar app aqui, pois pode causar "ReferenceError: . does not have a value"
  // As funções individuais verificarão se app está disponível quando necessário

  // ============================================
  // FUNÇÕES DE INFORMAÇÃO DA SEQUÊNCIA
  // ============================================

  /**
   * Obtém informações da sequência ativa
   * @returns {string} JSON string com informações da sequência
   */
  function getSequenceInfo() {
    try {
      var sequence = app.project.activeSequence;
      if (!sequence) {
        return JSON.stringify({
          success: false,
          error: 'Nenhuma sequência ativa'
        });
      }

      return JSON.stringify({
        success: true,
        id: sequence.sequenceID,
        name: sequence.name,
        timebase: sequence.timebase,
        duration: sequence.end ? sequence.end.seconds : 0,
        videoTracks: sequence.videoTracks ? sequence.videoTracks.numTracks : 0,
        audioTracks: sequence.audioTracks ? sequence.audioTracks.numTracks : 0
      });
    } catch (error) {
      return JSON.stringify({
        success: false,
        error: 'Erro ao obter informações da sequência: ' + error.toString()
      });
    }
  }

  /**
   * Obtém o tempo atual do playhead
   * @returns {string} JSON string com o tempo atual
   */
  function getPlayheadTime() {
    try {
      var sequence = app.project.activeSequence;
      if (!sequence) {
        return JSON.stringify({
          success: false,
          error: 'Nenhuma sequência ativa'
        });
      }

      var playhead = sequence.getPlayerPosition();
      return JSON.stringify({
        success: true,
        time: playhead.seconds,
        ticks: playhead.ticks
      });
    } catch (error) {
      return JSON.stringify({
        success: false,
        error: 'Erro ao obter tempo do playhead: ' + error.toString()
      });
    }
  }

  /**
   * Move o playhead para um tempo específico
   * @param {number} time - Tempo em segundos
   * @returns {string} JSON string com resultado
   */
  function movePlayheadToTime(time) {
    try {
      var sequence = app.project.activeSequence;
      if (!sequence) {
        return JSON.stringify({
          success: false,
          error: 'Nenhuma sequência ativa'
        });
      }

      var timeObj = new Time();
      timeObj.seconds = parseFloat(time);
      sequence.setPlayerPosition(timeObj);

      return JSON.stringify({
        success: true,
        time: timeObj.seconds
      });
    } catch (error) {
      return JSON.stringify({
        success: false,
        error: 'Erro ao mover playhead: ' + error.toString()
      });
    }
  }

  // ============================================
  // FUNÇÕES DE MARCADORES
  // ============================================

  /**
   * Obtém todos os marcadores da sequência ativa
   * @returns {string} JSON string com array de marcadores
   */
  function getMarkers() {
    try {
      var sequence = app.project.activeSequence;
      if (!sequence) {
        return JSON.stringify({
          success: false,
          error: 'Nenhuma sequência ativa',
          markers: []
        });
      }

      var markers = sequence.markers;
      if (!markers || markers.numMarkers === 0) {
        return JSON.stringify({
          success: true,
          markers: []
        });
      }

      var markersArray = [];
      var marker = markers.getFirstMarker();
      
      while (marker) {
        // Extrair ID do Flluir do nome do marcador (formato: "Autor [#ID]")
        var idMatch = marker.name.match(/\[#(.*?)\]/);
        var flluirId = idMatch ? idMatch[1] : null;

        markersArray.push({
          id: flluirId,
          name: marker.name,
          comments: marker.comments,
          start: marker.start.seconds,
          end: marker.end ? marker.end.seconds : marker.start.seconds,
          duration: marker.end ? (marker.end.seconds - marker.start.seconds) : 0,
          type: marker.type
        });

        marker = markers.getNextMarker(marker);
      }

      return JSON.stringify({
        success: true,
        markers: markersArray
      });
    } catch (error) {
      return JSON.stringify({
        success: false,
        error: 'Erro ao obter marcadores: ' + error.toString(),
        markers: []
      });
    }
  }

  /**
   * Remove todos os marcadores da sequência ativa
   * @returns {string} JSON string com resultado
   */
  function clearMarkers() {
    try {
      var sequence = app.project.activeSequence;
      if (!sequence) {
        return JSON.stringify({
          success: false,
          error: 'Nenhuma sequência ativa'
        });
      }

      var markers = sequence.markers;
      if (!markers || markers.numMarkers === 0) {
        return JSON.stringify({
          success: true,
          deleted: 0
        });
      }

      var deleted = 0;
      var marker = markers.getFirstMarker();
      while (marker) {
        var nextMarker = markers.getNextMarker(marker);
        try {
          markers.deleteMarker(marker);
          deleted++;
        } catch (e) {
          // Ignorar falhas pontuais e continuar
        }
        marker = nextMarker;
      }

      return JSON.stringify({
        success: true,
        deleted: deleted
      });
    } catch (error) {
      return JSON.stringify({
        success: false,
        error: 'Erro ao remover marcadores: ' + error.toString()
      });
    }
  }

  /**
   * Cria múltiplos marcadores a partir de um array de comentários
   * @param {string} commentsData - JSON string com array de comentários
   * @returns {string} JSON string com resultado
   */
  function createMarkers(commentsData) {
    try {
      // Verificar se app.project existe
      if (!app || !app.project) {
        return JSON.stringify({
          success: false,
          error: 'app.project não está disponível'
        });
      }

      var sequence = app.project.activeSequence;
      if (!sequence) {
        return JSON.stringify({
          success: false,
          error: 'Nenhuma sequência ativa. Abra uma sequência no Premiere Pro primeiro.'
        });
      }

      // Verificar se markers existe
      if (!sequence.markers) {
        return JSON.stringify({
          success: false,
          error: 'Sequência não possui coleção de marcadores (markers)'
        });
      }

      var comments;
      try {
        comments = JSON.parse(commentsData);
      } catch (parseError) {
        return JSON.stringify({
          success: false,
          error: 'Erro ao parsear JSON: ' + parseError.toString()
        });
      }

      if (!Array.isArray(comments)) {
        return JSON.stringify({
          success: false,
          error: 'commentsData não é um array. Tipo recebido: ' + typeof comments
        });
      }

      if (comments.length === 0) {
        return JSON.stringify({
          success: false,
          error: 'Array de comentários vazio'
        });
      }

      var markers = sequence.markers;
      var created = 0;
      var errors = [];

      for (var i = 0; i < comments.length; i++) {
        try {
          var comment = comments[i];
          
          // Validar estrutura do comentário
          if (!comment) {
            errors.push('Comentário ' + i + ': objeto nulo ou indefinido');
            continue;
          }

          var time = parseFloat(comment.timestamp || 0);
          
          if (!isFinite(time) || time < 0) {
            errors.push('Comentário ' + i + ' (ID: ' + (comment.id || 'N/A') + '): timestamp inválido (' + comment.timestamp + ')');
            continue;
          }
          
          // Criar marcador - createMarker espera um número (Float) em segundos, não um objeto Time
          var marker = markers.createMarker(time);
          if (!marker) {
            errors.push('Comentário ' + i + ' (ID: ' + (comment.id || 'N/A') + '): falha ao criar marcador');
            continue;
          }

          // Configurar nome do marcador
          var authorName = String(comment.author || 'Usuário');
          var markerName = authorName + ' [#' + String(comment.id || i) + ']';
          marker.name = markerName.substring(0, 255);
          
          // Configurar comentário do marcador
          var markerText = String(comment.text || '');
          marker.comments = markerText.substring(0, 1000);
          
          // Configurar duração se houver
          if (comment.duration && parseFloat(comment.duration) > 0) {
            try {
              var endTimeSeconds = time + parseFloat(comment.duration);
              var endTime = new Time();
              endTime.seconds = endTimeSeconds;
              marker.end = endTime;
            } catch (durationError) {
              // Não é crítico, apenas logar
              errors.push('Comentário ' + i + ': erro ao definir duração (não crítico)');
            }
          }

          created++;
        } catch (error) {
          var commentId = 'N/A';
          try {
            if (comments[i] && comments[i].id) {
              commentId = comments[i].id;
            }
          } catch (e) {
            // Ignorar
          }
          errors.push('Comentário ' + i + ' (ID: ' + commentId + '): ' + error.toString());
        }
      }

      return JSON.stringify({
        success: true,
        created: created,
        total: comments.length,
        errors: errors.length > 0 ? errors : null
      });
    } catch (error) {
      return JSON.stringify({
        success: false,
        error: 'Erro ao criar marcadores: ' + error.toString() + ' (tipo: ' + typeof error + ')'
      });
    }
  }

  /**
   * Move o playhead para um marcador específico
   * @param {string} markerId - ID do marcador (extraído do nome)
   * @returns {string} JSON string com resultado
   */
  function movePlayheadToMarker(markerId) {
    try {
      var sequence = app.project.activeSequence;
      if (!sequence) {
        return JSON.stringify({
          success: false,
          error: 'Nenhuma sequência ativa'
        });
      }

      var markers = sequence.markers;
      if (!markers || markers.numMarkers === 0) {
        return JSON.stringify({
          success: false,
          error: 'Nenhum marcador encontrado'
        });
      }

      var marker = markers.getFirstMarker();
      while (marker) {
        var idMatch = marker.name.match(/\[#(.*?)\]/);
        if (idMatch && idMatch[1] === markerId) {
          sequence.setPlayerPosition(marker.start);
          return JSON.stringify({
            success: true,
            time: marker.start.seconds
          });
        }
        marker = markers.getNextMarker(marker);
      }

      return JSON.stringify({
        success: false,
        error: 'Marcador não encontrado: ' + markerId
      });
    } catch (error) {
      return JSON.stringify({
        success: false,
        error: 'Erro ao mover playhead: ' + error.toString()
      });
    }
  }

  // ============================================
  // EXPORTAR FUNÇÕES PARA O HTML/JS
  // ============================================

  // Expor funções globalmente para serem chamadas via evalScript
  // IMPORTANTE: Em ExtendScript, definir $.global.FlluirSync de forma direta
  // Não usar verificações complexas que podem causar "ReferenceError: . does not have a value"
  
  // Criar objeto FlluirSync
  var flluirSyncObj = {
    getSequenceInfo: getSequenceInfo,
    getPlayheadTime: getPlayheadTime,
    movePlayheadToTime: movePlayheadToTime,
    getMarkers: getMarkers,
    createMarkers: createMarkers,
    clearMarkers: clearMarkers,
    movePlayheadToMarker: movePlayheadToMarker
  };
  
  // Definir em $.global de forma direta
  // Em ExtendScript, $ sempre existe e $.global é válido
  $.global = $.global || {};
  $.global.FlluirSync = flluirSyncObj;

})(this);
